#region References

using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Text;

using gov.va.med.vbecs.Common;

using ARTIFICIAL_COLUMN_NAMES = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;

using gov.va.med.vbecs.DAL.HL7.OpenLibrary;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>David Askew</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>11/16/2007</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// VbecsBceBpsMessage: used for communication from VBECS to COTS BCE software.  
	/// Provides information relating to assigned and/or issued blood products.
	/// CR 2962
	/// </summary>
	public sealed class VbecsBpsMessage
	{
		#region Inner Classes

		/// <summary>
		/// 
		/// </summary>
		private sealed class VbecsBpsMessageData
		{
			#region Variables

			private string _antibodyScreenResult;
			private string _bloodUnitGuid;
			private string _componentClassId;
			private string _componentClassName;
			private string _cprsOrderNumber;
			private string _crossmatchResult;
			private string _divisionCode;
			private string _eyeReadableUnitId;
			private string _issueToLocationName;
			private string _orderingProviderFirstName;
			private string _orderingProviderId;
			private string _orderingProviderLastName;
			private string _orderingProviderMiddleInitial;
			private string _patientABO;
			private string _patientDob;
			private string _patientFirstName;
			private string _patientGuid;
			private string _patientIcn;
			private string _patientLastName;
			private string _patientMiddleName;
			private string _patientRh;
			private string _patientSexCode;
			private string _patientNumber;
			private string _patientTransfusionRequirements;
			private string _requiredUnitQuantity;
			private string _treatingSpecialtyCode;
			private string _unitAboRh;
			private string _unitExpirationDate;
			private string _unitProductCode;
			private string _vistaPatientId;
		
			#endregion

			#region Properties

			/// <summary>
			/// 
			/// </summary>
			internal string AntibodyScreenResult
			{
				get
				{
					return _antibodyScreenResult;
				}
				set
				{
					_antibodyScreenResult = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string BloodUnitGuid
			{
				get
				{
					return _bloodUnitGuid;
				}
				set
				{
					_bloodUnitGuid = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string ComponentClassId
			{
				get
				{
					return _componentClassId;
				}
				set
				{
					_componentClassId = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string ComponentClassName
			{
				get
				{
					return _componentClassName;
				}
				set
				{
					_componentClassName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string CprsOrderNumber
			{
				get
				{
					return _cprsOrderNumber;
				}
				set
				{
					_cprsOrderNumber = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string DivisionCode
			{
				get
				{
					return _divisionCode;
				}
				set
				{
					_divisionCode = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string CrossmatchResult
			{
				get
				{
					return _crossmatchResult;
				}
				set
				{
					_crossmatchResult = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string EyeReadableUnitId
			{
				get
				{
					return _eyeReadableUnitId;
				}
				set
				{
					_eyeReadableUnitId = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string IssueToLocationName
			{
				get
				{
					return _issueToLocationName;
				}
				set
				{
					_issueToLocationName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string OrderingProviderFirstName
			{
				get
				{
					return _orderingProviderFirstName;
				}
				set
				{
					_orderingProviderFirstName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string OrderingProviderId
			{
				get
				{
					return _orderingProviderId;
				}
				set
				{
					_orderingProviderId = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string OrderingProviderLastName
			{
				get
				{
					return _orderingProviderLastName;
				}
				set
				{
					_orderingProviderLastName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string OrderingProviderMiddleInitial
			{
				get
				{
					return _orderingProviderMiddleInitial;
				}
				set
				{
					_orderingProviderMiddleInitial = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientABO
			{
				get
				{
					return _patientABO;
				}
				set
				{
					_patientABO = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientDob
			{
				get
				{
					return _patientDob;
				}
				set
				{
					_patientDob = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientFirstName
			{
				get
				{
					return _patientFirstName;
				}
				set
				{
					_patientFirstName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientGuid
			{
				get
				{
					return _patientGuid;
				}
				set
				{
					_patientGuid = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientIcn
			{
				get
				{
					return _patientIcn;
				}
				set
				{
					_patientIcn = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientLastName
			{
				get
				{
					return _patientLastName;
				}
				set
				{
					_patientLastName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientMiddleName
			{
				get
				{
					return _patientMiddleName;
				}
				set
				{
					_patientMiddleName = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientRh
			{
				get
				{
					return _patientRh;
				}
				set
				{
					_patientRh = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientSexCode
			{
				get
				{
					return _patientSexCode;
				}
				set
				{
					_patientSexCode = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientNumber
			{
				get
				{
					return _patientNumber;
				}
				set
				{
					_patientNumber = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string PatientTransfusionRequirements
			{
				get
				{
					return _patientTransfusionRequirements;
				}
				set
				{
					_patientTransfusionRequirements = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string RequiredUnitQuantity
			{
				get
				{
					return _requiredUnitQuantity;
				}
				set
				{
					_requiredUnitQuantity = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string TreatingSpecialtyCode
			{
				get
				{
					return _treatingSpecialtyCode;
				}
				set
				{
					_treatingSpecialtyCode = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string UnitAboRh
			{
				get
				{
					return _unitAboRh;
				}
				set
				{
					_unitAboRh = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string UnitExpirationDate
			{
				get
				{
					return _unitExpirationDate;
				}
				set
				{
					_unitExpirationDate = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string UnitProductCode
			{
				get
				{
					return _unitProductCode;
				}
				set
				{
					_unitProductCode = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string VistaPatientId
			{
				get
				{
					return _vistaPatientId;
				}
				set
				{
					_vistaPatientId = value;
				}
			}
	
			#endregion
		}

		#endregion

		private const string CARRIAGE_RETURN = "\x0D";
		private const string MESSAGE_TYPE = "BPS~O29";
		private const string ERROR_SAVING_HL7 = "ERROR ENCOUNTERED PROCESSING REQUEST\n\n\n The code location and error description are as follows: HL7AL.VbecsBpsMessage.ConstructAndSendMessage(): Unable to save new HL7 message in the MessageLog table.";
		private const string ERROR_RETRIEVING_VBECS_DATA = "ERROR ENCOUNTERED PROCESSING REQUEST\n\n\n The code location and error description are as follows: HL7AL.VbecsBpsMessage.ConstructAndSendMessage(): Unable to retrieve blood unit and patient data from the database.";
		private const string ECM = "Compatible - Electronically Crossmatched";

		#region Public Methods

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/12/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8536"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8537"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method will retrieve blood unit and corresponding patient data for all units
		/// currently in assigned, crossmatched or issued status.  For each such record, the
		/// method calls ConstructAndSendMessage() to insert the information into the 
		/// MessageLog table for pending transmission to the BCE COTS system.
		/// </summary>
		public static int BceCotsInterfaceActivationUpdate()
		{
			int recordsSent = 0;
			//
			// CR 2940: The BCE COTS interface is configurable in VBECS Admin
			if ( InterfaceControl.GetInterfaceActiveIndicator(Common.InterfaceName.BCE_COTS) == InterfaceActiveStatus.Active )
			{
				DataSet dsBloodUnitAndPatientData = Common.StoredProcedure.GetData( STOREDPROC.GetBloodUnitAndPatientGuids.StoredProcName );
				//
				if ( dsBloodUnitAndPatientData != null && dsBloodUnitAndPatientData.Tables != null && dsBloodUnitAndPatientData.Tables.Count == 1 
					&& dsBloodUnitAndPatientData.Tables[0].Rows != null && dsBloodUnitAndPatientData.Tables[0].Rows.Count > 0 )
				{
					int rowCount = dsBloodUnitAndPatientData.Tables[0].Rows.Count;
					//
					// CR 2945: adding support for transactions
					ArrayList spArray = new ArrayList(rowCount);
					ArrayList dtArray = new ArrayList(rowCount);
					//
					// CR 3015: create an instance of the interface here which we use for each record to construct the message.
					HL7Interface bceInterface = new HL7Interface( InterfaceName.BCE_COTS.ToString() );
					//
					for (int idx = 0; idx < rowCount; idx++)
					{
						DataRow drBloodUnitAndPatientData = dsBloodUnitAndPatientData.Tables[0].Rows[idx];
						//
						Guid bloodUnitGuid = (Guid)drBloodUnitAndPatientData[TABLES.BloodUnit.BloodUnitGuid];
						//
						Guid patientGuid = (Guid)drBloodUnitAndPatientData[TABLES.Patient.PatientGuid];
						//
						// CR 3015
						Guid orderedComponentGuid = (Guid)drBloodUnitAndPatientData[TABLES.OrderedComponent.OrderedComponentGuid];
						//
						// CR 3015
						string issueToLocationName = drBloodUnitAndPatientData[TABLES.IssuedUnit.IssueToLocationName].ToString();
						//
						HL7BPDispenseStatus dispenseStatus = HL7BPDispenseStatus.UK;
						//
						switch ( (string)drBloodUnitAndPatientData[TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode] )
						{
							case "S":
							case "C":
							{
								dispenseStatus = HL7BPDispenseStatus.RS;
								break;
							}
							case "I":
							{
								dispenseStatus = HL7BPDispenseStatus.DS;
								break;
							}
						}
						//
						if ( dispenseStatus != HL7BPDispenseStatus.UK )
						{
							// CR 2945: adding support for transactions
							spArray.Add(STOREDPROC.HL7InsertMessageLog.StoredProcName);	
							//
							// CR 3015
							dtArray.Add(ConstructMessage(bceInterface, bloodUnitGuid, patientGuid, orderedComponentGuid, issueToLocationName, dispenseStatus, false));
							//
							recordsSent += 1;
						}
					}
					//
					// CR 2945: adding support for transactions
					if (spArray.Count > 0 && dtArray.Count > 0)
					{
						new StoredProcedure().TransactionalGetValue(spArray, dtArray);
					}
				}
			}
			//
			return recordsSent;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8518"> 
		///		<ExpectedInput>BloodUnitGuid, PatientGuid, HL7BPDispenseStatus</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8519"> 
		///		<ExpectedInput>Empty BloodUnitGuid, Empty PatientGuid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8843"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8844"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8845"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method builds a BPS O29 message for given blood unit and corresponding 
		/// patient information, using the HL7BPDispenseStatus to determine the type of 
		/// update and returns a DataTable for insertion into the MessageLog table.
		/// CR 3015: added orderedComponentGuid and issueToLocationName
		/// </summary>
		/// <param name="bloodUnitGuid"><see cref="Guid"/></param>
		/// <param name="patientGuid"><see cref="Guid"/></param>
		/// <param name="orderedComponentGuid"><see cref="Guid"/></param>
		/// <param name="issueToLocationName"><see cref="string"/></param>
		/// <param name="dispenseStatus"><see cref="HL7BPDispenseStatus"/></param>
		/// <param name="electronicCrossmatch"><see cref="bool"/></param>
		public static DataTable ConstructMessage( 
			Guid bloodUnitGuid, Guid patientGuid, Guid orderedComponentGuid, string issueToLocationName, HL7BPDispenseStatus dispenseStatus, bool electronicCrossmatch )
		{ 
			#region Handle Invalid Parameters

			string exception = string.Empty;
			//
			if ( bloodUnitGuid == Guid.Empty )
			{
				exception = "bloodUnitGuid is not valid";
			}
			//
			if ( patientGuid == Guid.Empty )
			{
				exception = exception != string.Empty ? exception += " and " + "patientGuid is not valid" : "patientGuid is not valid";
			}
			//
			if ( exception != string.Empty )
			{
				throw new ArgumentException( exception );
			}

			#endregion
			//
			// CR 2940: The BCE COTS interface is configurable in VBECS Admin
			if ( InterfaceControl.GetInterfaceActiveIndicator(Common.InterfaceName.BCE_COTS) == InterfaceActiveStatus.Active )
			{
				HL7Interface bceInterface = new HL7Interface( InterfaceName.BCE_COTS.ToString() );
				//
				string hl7BpsMessage = ConstructHL7BpsMessage( bceInterface, bloodUnitGuid, patientGuid, orderedComponentGuid, issueToLocationName, dispenseStatus, electronicCrossmatch );
				//
				// CR 2945: adding support for transactions
				return HL7MessageLog.GetMessageDataForMessageLogInsert(hl7BpsMessage, MessageStatus.PendingTransmission, bceInterface, Common.UpdateFunction.HL7BceInterface);
			}
			//
			return new DataTable();
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8850"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8851"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method builds a BPS O29 message for given blood unit and corresponding 
		/// patient information, using the HL7BPDispenseStatus to determine the type 
		/// of update and returns a DataTable for insertion into the MessageLog table. 
		/// The pending message is then sent to the BCE COTS system.
		/// CR 2941
		/// CR 3015: added orderedComponentGuid and issueToLocationName
		/// </summary>
		/// <param name="bloodUnitGuid"><see cref="Guid"/></param>
		/// <param name="patientGuid"><see cref="Guid"/></param>
		/// <param name="orderedComponentGuid"><see cref="Guid"/></param>
		/// <param name="issueToLocationName"><see cref="string"/></param>
		/// <param name="dispenseStatus"><see cref="HL7BPDispenseStatus"/></param>
		public static void ConstructAndSendMessage( Guid bloodUnitGuid, Guid patientGuid, Guid orderedComponentGuid, string issueToLocationName, HL7BPDispenseStatus dispenseStatus )
		{
			#region Handle Invalid Parameters

			string exception = string.Empty;
			//
			if ( bloodUnitGuid == Guid.Empty )
			{
				exception = "bloodUnitGuid is not valid";
			}
			//
			if ( patientGuid == Guid.Empty )
			{
				exception = exception != string.Empty ? exception += " and " + "patientGuid is not valid" : "patientGuid is not valid";
			}
			//
			if ( exception != string.Empty )
			{
				throw new ArgumentException( exception );
			}

			#endregion
			//
			// CR 2940: The BCE COTS interface is configurable in VBECS Admin
			if ( InterfaceControl.GetInterfaceActiveIndicator(Common.InterfaceName.BCE_COTS) == InterfaceActiveStatus.Active )
			{
				HL7Interface bceInterface = new HL7Interface( InterfaceName.BCE_COTS.ToString() );
				//
				string hl7BpsMessage = ConstructHL7BpsMessage( bceInterface, bloodUnitGuid, patientGuid, orderedComponentGuid, issueToLocationName, dispenseStatus, false );
				//
				// File the message in the Message Log with a Pending Transmission status
				bool success = ( HL7MessageLog.InsertMessageLog( 
					HL7MessageLog.GetMessageDataForMessageLogInsert( 
					hl7BpsMessage, MessageStatus.PendingTransmission, bceInterface, Common.UpdateFunction.HL7BceInterface)) == 0 );
				//
				if ( !success )
				{
					throw new HL7.OpenLibrary.HL7Exception( ERROR_SAVING_HL7 );
				}
			}
		}

		#endregion

		#region Private Methods

		/// <summary>
		/// This method builds a BPS O29 message for given blood unit and corresponding 
		/// patient information, using the HL7BPDispenseStatus to determine the type of 
		/// update and returns a DataTable for insertion into the MessageLog table.
		/// CR 3015: This version is used by BceCotsInterfaceActivationUpdate() to avoid creating
		/// an interface object for each message that is constructed.  
		/// Note: interface activation status is not checked in this method; this method should
		/// only be called if the BCE interface is active.
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="issueToLocationName"></param>
		/// <param name="dispenseStatus"></param>
		/// <param name="electronicCrossmatch"></param>
		/// <returns></returns>
		private static DataTable ConstructMessage( 
			HL7Interface bceInterface, Guid bloodUnitGuid, Guid patientGuid, Guid orderedComponentGuid, string issueToLocationName, HL7BPDispenseStatus dispenseStatus, bool electronicCrossmatch )
		{ 
			#region Handle Invalid Parameters

			string exception = string.Empty;
			//
			if ( bceInterface == null )
			{
				exception = "bceInterface is not valid";
			}
			//
			if ( bloodUnitGuid == Guid.Empty )
			{
				exception = exception != string.Empty ? exception += " and " + "bloodUnitGuid is not valid" : "bloodUnitGuid is not valid";
			}
			//
			if ( patientGuid == Guid.Empty )
			{
				exception = exception != string.Empty ? exception += " and " + "patientGuid is not valid" : "patientGuid is not valid";
			}
			//
			if ( exception != string.Empty )
			{
				throw new ArgumentException( exception );
			}

			#endregion
			//
			string hl7BpsMessage = ConstructHL7BpsMessage( bceInterface, bloodUnitGuid, patientGuid, orderedComponentGuid, issueToLocationName, dispenseStatus, electronicCrossmatch );
			//
			// CR 2945: adding support for transactions
			return HL7MessageLog.GetMessageDataForMessageLogInsert(hl7BpsMessage, MessageStatus.PendingTransmission, bceInterface, Common.UpdateFunction.HL7BceInterface);
		}

		/// <summary>
		/// CR 3015
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="issueToLocationName"></param>
		/// <param name="dispenseStatus"></param>
		/// <param name="electronicCrossmatch"></param>
		/// <returns></returns>
		private static string ConstructHL7BpsMessage(
			HL7Interface bceInterface, Guid bloodUnitGuid, Guid patientGuid, Guid orderedComponentGuid, string issueToLocationName, HL7BPDispenseStatus dispenseStatus, bool electronicCrossmatch)
		{
			string errorMessage = string.Empty;
			//
			VbecsBpsMessageData vbecsBpsMessageData = 
				GetBloodUnitAndPatientDataForBce( bloodUnitGuid, patientGuid, orderedComponentGuid, issueToLocationName, bceInterface.EncodingCharacters[1] );
			//
			// CR 3015: set issue to location name
			vbecsBpsMessageData.IssueToLocationName = issueToLocationName;
			//
			PatientLocation patientLocation = DAL.HL7AL.PatientLocation.GetCurrentPatientLocation( patientGuid );
			//
			// CR 2966
			DateTime currentTime = DateTime.Now;
			string messageTime = currentTime.ToString("yyyyMMddHHmmsszzz").Replace(":",""); // CR 3084: changing to 24 hour time (HH)
			string messageControlId = currentTime.ToString("yMMddHHssffff"); // CR 3084: changing to 24 hour time (HH)
			//
			StringBuilder hl7BpsMessage = new StringBuilder();
			hl7BpsMessage.Append( BuildMshSegment(bceInterface, messageControlId, messageTime) );
			hl7BpsMessage.Append( BuildPidSegment(bceInterface, vbecsBpsMessageData) );
			hl7BpsMessage.Append( BuildPv1Segment(bceInterface, vbecsBpsMessageData, patientLocation) );
			hl7BpsMessage.Append( BuildOrcSegment(bceInterface, vbecsBpsMessageData) );
			hl7BpsMessage.Append( BuildBpoSegment(bceInterface, vbecsBpsMessageData) );
			hl7BpsMessage.Append( BuildBpxSegment(bceInterface, vbecsBpsMessageData, dispenseStatus, messageTime) );
			hl7BpsMessage.Append( BuildObxSegments(bceInterface, vbecsBpsMessageData, electronicCrossmatch) ); // CR 2945
			//
			return hl7BpsMessage.ToString();
		}

		/// <summary>
		/// This method retrieves pre-transfusion relevant data for given blood unit and patient,
		/// loading the retrieved data into a vbecsBpsMessageData object for use in constructing
		/// the BPS O29 HL7 message.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="issueToLocation"></param>
		/// <param name="transfusionRequirementsSeparator"></param>
		/// <returns></returns>
		private static VbecsBpsMessageData GetBloodUnitAndPatientDataForBce(
			Guid bloodUnitGuid, Guid patientGuid, Guid orderedComponentGuid, string issueToLocation, char transfusionRequirementsSeparator)
		{
			VbecsBpsMessageData vbecsBpsMessageData = null;
			//
			SqlParameter [] prms =
			{
				new SqlParameter(STOREDPROC.GetBloodUnitAndPatientData.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitAndPatientData.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitAndPatientData.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier)
			};
			//
			prms[0].Value = patientGuid;
			prms[1].Value = bloodUnitGuid;
			prms[2].Value = orderedComponentGuid;
			//
			DataSet dsBceData = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitAndPatientData.StoredProcName, prms);
			//
			if (dsBceData != null && dsBceData.Tables != null && dsBceData.Tables.Count == 4
				&& dsBceData.Tables[0].Rows != null && dsBceData.Tables[0].Rows.Count == 1)
			{
				string transfusionRequirements = string.Empty;
				//
				int transfusionRequirementsRowCount = dsBceData.Tables[3] != null ? dsBceData.Tables[3].Rows.Count : 0;
				if (transfusionRequirementsRowCount > 0)
				{
					DataTable dtTransfusionRequirements = dsBceData.Tables[3];
					StringBuilder transfusionRequirementBuilder = new StringBuilder();
					ArrayList uniqueRequirements = new ArrayList();
					//
					for (int idx = 0; idx < transfusionRequirementsRowCount; idx++)
					{
						string requirement = Convert.ToString(dtTransfusionRequirements.Rows[idx][ARTIFICIAL_COLUMN_NAMES.PatientTransfusionRequirement]).Trim();
						//
						// Make sure we only add each requirement once; more than one requirement 
						// could have the same abbreviation ( ex. WSH for Washed PL   n  Washed RBC)
						if ( !uniqueRequirements.Contains(requirement) )
						{
							transfusionRequirementBuilder.Append( requirement );
							transfusionRequirementBuilder.Append( transfusionRequirementsSeparator );
							uniqueRequirements.Add( requirement );
						}
					}
					//
					int transfusionRequirementBuilderLength = transfusionRequirementBuilder.Length;
					if (transfusionRequirementBuilderLength > 0)
					{
						// remove the final repetition separator
						transfusionRequirementBuilder.Remove(transfusionRequirementBuilderLength - 1, 1);
					}
					//
					transfusionRequirements = transfusionRequirementBuilder.ToString();
				}
				//
				vbecsBpsMessageData = new VbecsBpsMessageData();
				//
				// Patient Data  
				vbecsBpsMessageData.PatientNumber = dsBceData.Tables[0].Rows[0][TABLES.Patient.PatientSsn].ToString();
				vbecsBpsMessageData.VistaPatientId = dsBceData.Tables[0].Rows[0][TABLES.Patient.VistaPatientId].ToString();
				vbecsBpsMessageData.PatientIcn = dsBceData.Tables[0].Rows[0][TABLES.Patient.PatientIcn].ToString();
				vbecsBpsMessageData.PatientLastName = dsBceData.Tables[0].Rows[0][TABLES.Patient.PatientLastName].ToString();
				vbecsBpsMessageData.PatientFirstName = dsBceData.Tables[0].Rows[0][TABLES.Patient.PatientFirstName].ToString();
				vbecsBpsMessageData.PatientMiddleName = dsBceData.Tables[0].Rows[0][TABLES.Patient.PatientMiddleName].ToString();
				//**************************************************************************************************************************
				// CR 3186: Patient Dob and Sex Code can be null
				vbecsBpsMessageData.PatientDob = !dsBceData.Tables[0].Rows[0].IsNull(TABLES.Patient.PatientDob) ? 
					HL7DateFormat.ConvertDateTimeToHL7((DateTime)dsBceData.Tables[0].Rows[0][TABLES.Patient.PatientDob]) : string.Empty; 
				vbecsBpsMessageData.PatientSexCode = !dsBceData.Tables[0].Rows[0].IsNull(TABLES.Patient.PatientSexCode) ? 
					dsBceData.Tables[0].Rows[0][TABLES.Patient.PatientSexCode].ToString() : string.Empty;
				//**************************************************************************************************************************
				vbecsBpsMessageData.PatientABO = dsBceData.Tables[0].Rows[0][ARTIFICIAL_COLUMN_NAMES.PatientABO].ToString();
				vbecsBpsMessageData.PatientRh = dsBceData.Tables[0].Rows[0][ARTIFICIAL_COLUMN_NAMES.PatientRh].ToString().ToUpper();
				vbecsBpsMessageData.AntibodyScreenResult = dsBceData.Tables[0].Rows[0][TABLES.TestResult.TestResultText].ToString();
				// Blood Unit Data
				vbecsBpsMessageData.BloodUnitGuid = bloodUnitGuid.ToString();
				vbecsBpsMessageData.DivisionCode = dsBceData.Tables[1].Rows[0][TABLES.BloodUnit.DivisionCode].ToString();
				vbecsBpsMessageData.EyeReadableUnitId = dsBceData.Tables[1].Rows[0][TABLES.BloodUnit.EyeReadableUnitId].ToString();
				vbecsBpsMessageData.ComponentClassId = dsBceData.Tables[1].Rows[0][TABLES.ComponentClass.ComponentClassId].ToString();
				vbecsBpsMessageData.ComponentClassName = dsBceData.Tables[1].Rows[0][TABLES.ComponentClass.ComponentClassName].ToString();
				vbecsBpsMessageData.UnitProductCode = dsBceData.Tables[1].Rows[0][ARTIFICIAL_COLUMN_NAMES.UnitProductCode].ToString().Trim();
				// CR 3084: changing to 24 hour time (HH)
                // Defect 227795
                DateTime unitExpirationDate = (DateTime)dsBceData.Tables[1].Rows[0][TABLES.BloodUnitMedia.UnitExpirationDate];
                vbecsBpsMessageData.UnitExpirationDate = DAL.VbecsDateTime.AddDivisionTimeZoneForHL7Messages(unitExpirationDate, vbecsBpsMessageData.DivisionCode);

				vbecsBpsMessageData.UnitAboRh = dsBceData.Tables[1].Rows[0][ARTIFICIAL_COLUMN_NAMES.UnitAboRh].ToString().Trim();
				//
				// Other Data
				vbecsBpsMessageData.TreatingSpecialtyCode = dsBceData.Tables[2].Rows[0][TABLES.PatientTreatment.TreatingSpecialtyCode].ToString();
				vbecsBpsMessageData.RequiredUnitQuantity = dsBceData.Tables[2].Rows[0][TABLES.OrderedComponent.RequiredUnitQuantity].ToString();
				vbecsBpsMessageData.CprsOrderNumber = dsBceData.Tables[2].Rows[0][TABLES.OrderedComponent.CprsOrderNumber].ToString();
				vbecsBpsMessageData.OrderingProviderId = dsBceData.Tables[2].Rows[0][TABLES.PatientOrder.OrderingProviderId].ToString();
				vbecsBpsMessageData.OrderingProviderLastName = dsBceData.Tables[2].Rows[0][TABLES.PatientOrder.OrderingProviderLastName].ToString();
				vbecsBpsMessageData.OrderingProviderFirstName = dsBceData.Tables[2].Rows[0][TABLES.PatientOrder.OrderingProviderFirstName].ToString();
				vbecsBpsMessageData.OrderingProviderMiddleInitial = dsBceData.Tables[2].Rows[0][TABLES.PatientOrder.OrderingProviderMiddleInitial].ToString();
				vbecsBpsMessageData.CrossmatchResult = dsBceData.Tables[2].Rows[0][TABLES.CrossmatchResult.CrossmatchResultText].ToString();
				//
				// Transfusion Requirements (if any)
				vbecsBpsMessageData.PatientTransfusionRequirements = transfusionRequirements;
			}
			else
			{
				throw new HL7.OpenLibrary.HL7Exception( ERROR_RETRIEVING_VBECS_DATA );
			}
			//
			return vbecsBpsMessageData;
		}

		/// <summary>
		/// This method builds the Msh Segment
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="messageControlId"></param>
		/// <param name="messageTime"></param>
		/// <returns></returns>
		private static string BuildMshSegment(HL7Interface bceInterface, string messageControlId, string messageTime )
		{
			StringBuilder msh = new StringBuilder();
			//
			// MSH Segment ID
			msh.Append("MSH");																
			msh.Append(bceInterface.FieldSeparator);										
			msh.Append(bceInterface.EncodingCharacters);									
			msh.Append(bceInterface.FieldSeparator);
			// Sending Application
			msh.Append(bceInterface.VbecsApplicationId.Trim());							
			msh.Append(bceInterface.FieldSeparator);
			if( bceInterface.VbecsFacilityId != null )
			{	
				// Sending Facility
				msh.Append(bceInterface.VbecsFacilityId.Trim());							
			}
			msh.Append(bceInterface.FieldSeparator);
			// Receiving Application
			msh.Append(bceInterface.InterfaceApplicationId.Trim());						
			msh.Append(bceInterface.FieldSeparator);
			if( bceInterface.InterfaceFacilityId != null )
			{
				// Receiving Facility
				msh.Append(bceInterface.InterfaceFacilityId.Trim());						
			}
			msh.Append(bceInterface.FieldSeparator);
			msh.Append(messageTime);														
			msh.Append(bceInterface.FieldSeparator);
			msh.Append(bceInterface.FieldSeparator);
			msh.Append(MESSAGE_TYPE);														
			msh.Append(bceInterface.FieldSeparator);
			// Message Control ID
			msh.Append("VBECS").Append(messageControlId);		
			msh.Append(bceInterface.FieldSeparator);
			msh.Append(bceInterface.ProcessingId);											
			msh.Append(bceInterface.FieldSeparator);
			msh.Append(bceInterface.VersionId.Trim());										
			msh.Append(bceInterface.FieldSeparator);
			msh.Append(bceInterface.FieldSeparator);
			msh.Append(bceInterface.FieldSeparator);
			// Accept Acknowledgment Type
			msh.Append("AL");																
			msh.Append(bceInterface.FieldSeparator);
			// Application Acknowledgment Type
			msh.Append("AL");																
			msh.Append(CARRIAGE_RETURN);
			//
			return msh.ToString();
		}

		/// <summary>
		/// This method builds the Pid Segment
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="vbecsBpsMessageData"></param>
		/// <returns></returns>
		private static string BuildPidSegment(HL7Interface bceInterface, VbecsBpsMessageData vbecsBpsMessageData)
		{
			StringBuilder pid = new StringBuilder();
			//  
			pid.Append("PID");
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(vbecsBpsMessageData.PatientIcn);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(vbecsBpsMessageData.VistaPatientId);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(vbecsBpsMessageData.PatientLastName);
			pid.Append(bceInterface.EncodingCharacters[0]);
			pid.Append(vbecsBpsMessageData.PatientFirstName);
			pid.Append(bceInterface.EncodingCharacters[0]);
			pid.Append(vbecsBpsMessageData.PatientMiddleName);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(vbecsBpsMessageData.PatientDob);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(vbecsBpsMessageData.PatientSexCode);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(bceInterface.FieldSeparator);
			pid.Append(vbecsBpsMessageData.PatientNumber);
			pid.Append(CARRIAGE_RETURN);
			//
			return pid.ToString();
		}

		/// <summary>
		/// This method builds the Pv1 Segment
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="vbecsBpsMessageData"></param>
		/// <param name="patientLocation"></param>
		/// <returns></returns>
		private static string BuildPv1Segment(HL7Interface bceInterface, VbecsBpsMessageData vbecsBpsMessageData, PatientLocation patientLocation)
		{
			StringBuilder pv1 = new StringBuilder();
			//
			pv1.Append("PV1");
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(patientLocation.InPatientIndicator ? "I" : "O");		// PV1.2
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(patientLocation.Name);								// PV1.3.1
			pv1.Append(bceInterface.EncodingCharacters[0]);
			// CR 2970: Room and Bed are now stored in one field
			pv1.Append(patientLocation.RoomBed);							// PV1.3.2
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(patientLocation.Ien);								// PV1.3.4.1
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(bceInterface.EncodingCharacters[0]);
			pv1.Append(patientLocation.DivisionCode);						// PV1.3.10.1		
			pv1.Append(bceInterface.EncodingCharacters[3]);
			pv1.Append(patientLocation.DivisionName);						// PV1.3.10.2
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(bceInterface.FieldSeparator);
			pv1.Append(vbecsBpsMessageData.TreatingSpecialtyCode);			// PV1.10
			pv1.Append(CARRIAGE_RETURN);
			//
			return pv1.ToString();
		}

		/// <summary>
		/// This method builds the Orc Segment
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="vbecsBpsMessageData"></param>
		/// <returns></returns>
		private static string BuildOrcSegment(HL7Interface bceInterface, VbecsBpsMessageData vbecsBpsMessageData)
		{
			StringBuilder orc = new StringBuilder();
			//
			orc.Append("ORC");
			orc.Append(bceInterface.FieldSeparator);
			orc.Append("SC");								// Always send a Status Change Order Control code			
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(vbecsBpsMessageData.CprsOrderNumber);
			orc.Append(bceInterface.EncodingCharacters[0]);
			orc.Append("OR");
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(vbecsBpsMessageData.OrderingProviderId);
			orc.Append(bceInterface.EncodingCharacters[0]);
			orc.Append(vbecsBpsMessageData.OrderingProviderLastName);
			orc.Append(bceInterface.EncodingCharacters[0]);
			orc.Append(vbecsBpsMessageData.OrderingProviderFirstName);
			orc.Append(bceInterface.EncodingCharacters[0]);
			orc.Append(vbecsBpsMessageData.OrderingProviderMiddleInitial);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(bceInterface.FieldSeparator);
			orc.Append(vbecsBpsMessageData.DivisionCode);
			orc.Append(CARRIAGE_RETURN);
			//
			return orc.ToString();
		}

		/// <summary>
		/// This method builds the Bpo Segment
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="vbecsBpsMessageData"></param>
		/// <returns></returns>
		private static string BuildBpoSegment(HL7Interface bceInterface, VbecsBpsMessageData vbecsBpsMessageData)
		{
			StringBuilder bpo = new StringBuilder();
			//
			bpo.Append("BPO");
			bpo.Append(bceInterface.FieldSeparator);
			bpo.Append("1");
			bpo.Append(bceInterface.FieldSeparator);
			bpo.Append(vbecsBpsMessageData.ComponentClassId);
			bpo.Append(bceInterface.EncodingCharacters[0]);
			bpo.Append(vbecsBpsMessageData.ComponentClassName);
			bpo.Append(bceInterface.FieldSeparator);
			bpo.Append(bceInterface.EncodingCharacters[0]);
			bpo.Append(vbecsBpsMessageData.PatientTransfusionRequirements);
			bpo.Append(bceInterface.FieldSeparator);
			bpo.Append(vbecsBpsMessageData.RequiredUnitQuantity);
			bpo.Append(CARRIAGE_RETURN);
			//
			return bpo.ToString();
		}

		/// <summary>
		/// This method builds the Bpx Segment
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="vbecsBpsMessageData"></param>
		/// <param name="dispenseStatus"></param>
		/// <param name="messageTime"></param>
		/// <returns></returns>
		private static string BuildBpxSegment( 
			HL7Interface bceInterface, VbecsBpsMessageData vbecsBpsMessageData, HL7BPDispenseStatus dispenseStatus, string messageTime )
		{
			StringBuilder bpx = new StringBuilder();
			//
			bpx.Append("BPX");
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append("1");
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(Common.Utility.GetHL7BPDispenseStatusStringFromEnum(dispenseStatus));
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append("P");
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(messageTime);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(vbecsBpsMessageData.EyeReadableUnitId);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(vbecsBpsMessageData.UnitProductCode);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append("ABORH");
			bpx.Append(bceInterface.EncodingCharacters[0]);
			bpx.Append(vbecsBpsMessageData.UnitAboRh);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(vbecsBpsMessageData.UnitExpirationDate);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(vbecsBpsMessageData.RequiredUnitQuantity);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(bceInterface.FieldSeparator);
			bpx.Append(vbecsBpsMessageData.BloodUnitGuid);
			bpx.Append(bceInterface.FieldSeparator);
			//
			if ( dispenseStatus == Common.HL7BPDispenseStatus.DS )
			{
				bpx.Append(vbecsBpsMessageData.IssueToLocationName);
			}
			//
			bpx.Append(CARRIAGE_RETURN);
			//
			return bpx.ToString();
		}

		/// <summary>
		/// This method builds Obx Segment(s);
		/// may be 0 - 4
		/// </summary>
		/// <param name="bceInterface"></param>
		/// <param name="vbecsBpsMessageData"></param>
		/// <param name="electronicCrossmatch"></param>
		/// <returns></returns>
		private static string BuildObxSegments( HL7Interface bceInterface, VbecsBpsMessageData vbecsBpsMessageData, bool electronicCrossmatch )
		{
			StringBuilder obx = new StringBuilder();
			//
			// CR 2941
			if (vbecsBpsMessageData.CrossmatchResult != string.Empty || electronicCrossmatch)
			{
				obx.Append("OBX");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("CE");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("CMR");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				//
				// CR 2941
				if (electronicCrossmatch)
				{
					obx.Append(ECM);
				}
				else
				{
					obx.Append(vbecsBpsMessageData.CrossmatchResult);
				}
				//
				obx.Append(CARRIAGE_RETURN);
			}
			//
			if (vbecsBpsMessageData.AntibodyScreenResult != string.Empty)
			{
				obx.Append("OBX");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("CE");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("ASR");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(vbecsBpsMessageData.AntibodyScreenResult);
				obx.Append(CARRIAGE_RETURN);
			}
			//
			if (vbecsBpsMessageData.PatientABO != string.Empty)
			{
				obx.Append("OBX");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("CE");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("ABO");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(vbecsBpsMessageData.PatientABO);
				obx.Append(CARRIAGE_RETURN);
			}
			//
			if (vbecsBpsMessageData.PatientRh != string.Empty)
			{
				obx.Append("OBX");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("CE");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append("Rh");
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(bceInterface.FieldSeparator);
				obx.Append(vbecsBpsMessageData.PatientRh);
				obx.Append(CARRIAGE_RETURN);
			}
			//
			return obx.ToString();
		}

		#endregion
	}
}
